/*
 * SAXLib.h
 *
 * public header for XML decoding library
 *
 * Copyright (c) 2004-2005 inDev Software.
 * All rights reserved.
 */
 
#ifndef SAXLIB_H_
#define SAXLIB_H_

/* Palm OS common definitions */
#include <SystemMgr.h>

/* If we're actually compiling the library code, then we need to
 * eliminate the trap glue that would otherwise be generated from
 * this header file in order to prevent compiler errors in CW Pro 2. */
#ifdef BUILDING_SAXLIB
	#define SAXLIB_LIB_TRAP(trapNum)
#else
	#define SAXLIB_LIB_TRAP(trapNum) SYS_TRAP(trapNum)
#endif

/* Current library version xyyz (x - major version, yy - update, z - bugfix) */
#define		SAXLibVersion	0x1000
/* SAXLib type and creator */
#define		SAXLibCreatorID	'SAXl'
#define		SAXLibTypeID		sysFileTLibrary
/* Internal library name which can be passed to SysLibFind() */
#define		SAXLibName		"SAXLib"

/* SAXLib result codes */

/* invalid parameter */
#define SAXLibErrParam			(appErrorClass | 1)	
/* library is not open */
#define SAXLibErrNotOpen		(appErrorClass | 2)
/* returned from SAXLibClose() if the library is still open */
#define SAXLibErrStillOpen		(appErrorClass | 3)
/* returned when FileProvider interface is not set */
#define SAXLibErrNoFileProvider		(appErrorClass | 4)
/* file provider error */
#define SAXLibErrFileProviderError	(appErrorClass | 5)
/* not enough memory */
#define SAXLibErrNotEnoughMemory	(appErrorClass | 6)
/* incorrect parsing rule */
#define SAXLibErrIncorrectRule		(appErrorClass | 7)
/* unexpected end of file */
#define SAXLibErrUnexpectedEOF		(appErrorClass | 8)
/* rule does not work (returned when there is no rule signature at this point) */
#define SAXLibErrRuleDoesNotWork	(appErrorClass | 9)	// private!!! TODO: Move to another error set
/* file format is invalid */
#define SAXLibErrInvalidFile		(appErrorClass | 10)
/* file buffer management error */
#define SAXLibErrFileBuffer			(appErrorClass | 11)
/* XML file characters encoding not supported */
#define SAXLibErrEncodingNotSupported	(appErrorClass | 12)
/* Return when processing was interrupted by Content Handler */
#define SAXLibErrInterrupted		(appErrorClass | 13)

/*********************************************************************
 * Common type definitions
 *********************************************************************/
/* Supported output character encodings (with corresponding CharEncodingType value in comments)*/
typedef enum
{
	encodingCP1250=0,	/* charEncodingCP1250 */
	encodingCP1251,		/* charEncodingCP1251 */
	encodingCP1252,		/* charEncodingCP1252 */
	encodingCP1253,		/* charEncodingCP1253 */
	encodingCP1254,		/* charEncodingCP1254 */
	encodingCP1255,		/* charEncodingCP1255 */
	encodingCP1256,		/* charEncodingCP1256 */
	encodingCP1257,		/* charEncodingCP1257 */
	encodingISO8859_1,	/* charEncodingISO8859_1 */
	encodingISO8859_2,	/* charEncodingISO8859_2 */
	encodingISO8859_3,	/* charEncodingISO8859_3 */
	encodingISO8859_4,	/* charEncodingISO8859_4 */
	encodingISO8859_5,	/* charEncodingISO8859_5 */
	encodingISO8859_6,	/* charEncodingISO8859_6 */
	encodingISO8859_7,	/* charEncodingISO8859_7 */
	encodingISO8859_8,	/* charEncodingISO8859_8 */
	encodingISO8859_9	/* charEncodingISO8859_9 */
} CharacterEncoding;
 
/* ---------------------------------*
 * Content Handler "interface" 		*
 *----------------------------------*/
typedef void CH_StartDocumentFunc(); /* Receive notification of the beginning of a document */
typedef void CH_EndDocumentFunc(); /* Receive notification of the end of a document */
typedef void CH_StartElementFunc(Char* uri,Char* localName,Char* qName,UInt32 attrID); /* Receive notification of the beginning of an element */
typedef void CH_EndElementFunc(Char* uri,Char* localName,Char* qName); /* Receive notification of the end of an element */
typedef void CH_StartPrefixMappingFunc(Char* prefix,Char* uri); /* Begin the scope of a prefix-URI Namespace mapping */
typedef void CH_EndPrefixMappingFunc(Char* uri); /* End the scope of a prefix-URI mapping */
typedef Boolean CH_CharactersFunc(Char* ch,Int16 start,Int16 length,UInt32 currentState); /* Receive notification of character data, if returns false -- processing should be interrupted */
typedef void CH_IgnorableWhitespaceFunc(Char* ch,Int16 start,Int16 length); /* Receive notification of ignorable whitespace in element content */
typedef void CH_SkippedEntityFunc(Char* name); /* Receive notification of a skipped entity */
typedef void CH_ProcessingInstructionFunc(Char* target,Char* data); /* Receive notification of a processing instruction */
typedef void CH_SetDocumentLocatorFunc(UInt32 locatorID); /* Receive an object for locating the origin of SAX document events */
typedef struct ContentHandler
{
	CH_StartDocumentFunc			*StartDocument;
	CH_EndDocumentFunc				*EndDocument;
	CH_StartElementFunc				*StartElement;
	CH_EndElementFunc				*EndElement;
	CH_StartPrefixMappingFunc		*StartPrefixMapping;
	CH_EndPrefixMappingFunc			*EndPrefixMapping;
	CH_CharactersFunc				*Characters;
	CH_IgnorableWhitespaceFunc		*IgnorableWhitespace;
	CH_SkippedEntityFunc			*SkippedEntity;
	CH_ProcessingInstructionFunc	*ProcessingInstruction;
	CH_SetDocumentLocatorFunc		*SetDocumentLocator;
} ContentHandler;

/* ---------------------------------*
 * File Provider "interface" 		*
 * 									*
 * This "interface" should be set	*
 * before starting any parsing.		*
 * All functions are mandatory.		*
 *----------------------------------*/
typedef Boolean FP_OpenFileFunc(Char* uri,UInt32 *pFileRef); /* Opens the file */
typedef UInt16 FP_GetFileBlockSizeFunc(UInt32 fileRef); /* Gets file block size */
typedef Boolean FP_ReadFileFunc(UInt32 fileRef,void *pBuf,UInt16 *readBytes); /* Reads the next file block contents to the buffer */
typedef void FP_CloseFileFunc(UInt32 fileRef); /* Closes the opened file */
typedef Boolean FP_SeekFileFunc(UInt32 fileRef,Int32 *offset); /* Seeks to the specific position in the file */
typedef UInt32 FP_GetFileSizeFunc(UInt32 fileRef); /* Returns the file size */
typedef struct FileProvider
{
	FP_OpenFileFunc				*OpenFile;
	FP_GetFileBlockSizeFunc		*GetFileBlockSize;
	FP_ReadFileFunc				*ReadFile;
	FP_CloseFileFunc			*CloseFile;
	FP_SeekFileFunc				*SeekFile;
	FP_GetFileSizeFunc			*GetFileSize;
} FileProvider;
/*********************************************************************
 * API Prototypes
 *********************************************************************/
#ifdef __cplusplus
extern "C" {
#endif

/* Standard library open, close, sleep and wake functions */
extern Err SAXLibOpen(UInt16 refNum,UInt32 *pClientContext,UInt16 *pLibVersion)
	SAXLIB_LIB_TRAP(sysLibTrapOpen);				
extern Err SAXLibClose(UInt16 refNum,UInt32 clientContext)
	SAXLIB_LIB_TRAP(sysLibTrapClose);
extern Err SAXLibSleep(UInt16 refNum)
	SAXLIB_LIB_TRAP(sysLibTrapSleep);
extern Err SAXLibWake(UInt16 refNum)
	SAXLIB_LIB_TRAP(sysLibTrapWake);

/* Custom library API functions */

/* Sets ContentHandler "interface" */
extern void SAXLibSetContentHandler(UInt16 refNum,UInt32 clientContext,ContentHandler *handler)
	SAXLIB_LIB_TRAP(sysLibTrapBase + 5);	
/* Sets FileProvider "interface" */
extern void SAXLibSetFileProvider(UInt16 refNum,UInt32 clientContext,FileProvider *provider)
	SAXLIB_LIB_TRAP(sysLibTrapBase + 6);
/* Starts parsing from the file beginning */
extern Err SAXLibParse(UInt16 refNum, UInt32 clientContext,Char* uri,CharacterEncoding dstEncoding)
	SAXLIB_LIB_TRAP(sysLibTrapBase + 7);
/* Returns all information about current parsing state in one memory chunk */
extern Err SAXLibGetCurrentState(UInt16 refNum,UInt32 currentState,MemHandle *phState)
	SAXLIB_LIB_TRAP(sysLibTrapBase + 8);
/* Resumes the parsing from the point returned from SAXLibGetCurrentState */
extern Err SAXLibParseResume(UInt16 refNum,UInt32 clientContext,Char* uri,CharacterEncoding dstEncoding,MemHandle hState)
	SAXLIB_LIB_TRAP(sysLibTrapBase + 9);

/* Document Locator "interface" */
extern Char* SAXLib_DL_GetSystemId(UInt16 refNum,UInt32 locatorID) SAXLIB_LIB_TRAP(sysLibTrapBase+10);
extern Char* SAXLib_DL_GetPublicId(UInt16 refNum,UInt32 locatorID) SAXLIB_LIB_TRAP(sysLibTrapBase+11);
extern UInt16 SAXLib_DL_GetColumnNumber(UInt16 refNum,UInt32 locatorID) SAXLIB_LIB_TRAP(sysLibTrapBase+12);
extern UInt16 SAXLib_DL_GetLineNumber(UInt16 refNum,UInt32 locatorID) SAXLIB_LIB_TRAP(sysLibTrapBase+13);

/************************** 
 * Attributes "interface" *
 **************************/
/* Look up the index of an attribute by XML 1.0 qualified name. */
extern UInt16 SAXLib_Attr_GetIndexByQName(UInt16 refNum,UInt32 attrID,Char* qName) SAXLIB_LIB_TRAP(sysLibTrapBase+14);
/* Look up the index of an attribute by Namespace name. */
extern UInt16 SAXLib_Attr_GetIndexByNamespace(UInt16 refNum,UInt32 attrID,Char* uri,Char* localName) 
	SAXLIB_LIB_TRAP(sysLibTrapBase+15);
/* Return the number of attributes in the list. */
extern UInt16 SAXLib_Attr_GetLength(UInt16 refNum,UInt32 attrID) SAXLIB_LIB_TRAP(sysLibTrapBase+16);
/* Look up an attribute's local name by index. */
extern Char* SAXLib_Attr_GetLocalName(UInt16 refNum,UInt32 attrID,UInt16 index) SAXLIB_LIB_TRAP(sysLibTrapBase+17);
/* Look up an attribute's XML 1.0 qualified name by index. */
extern Char* SAXLib_Attr_GetQName(UInt16 refNum,UInt32 attrID,UInt16 index) SAXLIB_LIB_TRAP(sysLibTrapBase+18);
/* Look up an attribute's type by index. */
extern Char* SAXLib_Attr_GetTypeByIndex(UInt16 refNum,UInt32 attrID,UInt16 index) SAXLIB_LIB_TRAP(sysLibTrapBase+19);
/* Look up an attribute's type by XML 1.0 qualified name. */
extern Char* SAXLib_Attr_GetTypeByQName(UInt16 refNum,UInt32 attrID,Char* qName) SAXLIB_LIB_TRAP(sysLibTrapBase+20);
/* Look up an attribute's type by Namespace name. */
extern Char* SAXLib_Attr_GetTypeByNamespace(UInt16 refNum,UInt32 attrID,Char* uri,Char* localName)
	SAXLIB_LIB_TRAP(sysLibTrapBase+21);
/* Look up an attribute's Namespace URI by index. */
extern Char* SAXLib_Attr_GetUri(UInt16 refNum,UInt32 attrID,UInt16 index) SAXLIB_LIB_TRAP(sysLibTrapBase+22);
/* Look up an attribute's value by index. */
extern Char* SAXLib_Attr_GetValueByIndex(UInt16 refNum,UInt32 attrID,UInt16 index) SAXLIB_LIB_TRAP(sysLibTrapBase+23);
/* Look up an attribute's value by XML 1.0 qualified name. */
extern Char* SAXLib_Attr_GetValueByQName(UInt16 refNum,UInt32 attrID,Char* qName) SAXLIB_LIB_TRAP(sysLibTrapBase+24);
/* Look up an attribute's value by Namespace name. */
extern Char* SAXLib_Attr_GetValueByNamespace(UInt16 refNum,UInt32 attrID,Char* uri,Char* localName)
	SAXLIB_LIB_TRAP(sysLibTrapBase+25);

#ifdef __cplusplus
}
#endif

#endif /* SAXLIB_H_ */